// Copyright (c) 2000-2010, Heiko Bauke
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 
//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.  
// 
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.  
// 
//   * Neither the name of the copyright holder nor the names of its
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
// FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
// COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.

#if !(defined TRNG_UNIFORM01_DIST_HPP)

#define TRNG_UNIFORM01_DIST_HPP

#include <trng/limits.hpp>
#include <trng/utility.hpp>
#include <ostream>
#include <istream>
#include <cerrno>

namespace trng {

  // uniform random number generator class
  template<typename float_t=double>
  class uniform01_dist {
  public:
    typedef float_t result_type;
    class param_type;
    
    class param_type {
    public:
      param_type() {
      }

      friend class uniform01_dist<float_t>;

      // Streamable concept
      template<typename char_t, typename traits_t>
      friend std::basic_ostream<char_t, traits_t> &
      operator<<(std::basic_ostream<char_t, traits_t> &out,
		 const param_type &p) {
	std::ios_base::fmtflags flags(out.flags());
	out.flags(std::ios_base::dec | std::ios_base::fixed |
		  std::ios_base::left);
	out << '('
	    << ')';
	out.flags(flags);
	return out;
      }
      
      template<typename char_t, typename traits_t>
      friend std::basic_istream<char_t, traits_t> &
      operator>>(std::basic_istream<char_t, traits_t> &in,
		 param_type &p) {
	std::ios_base::fmtflags flags(in.flags());
	in.flags(std::ios_base::dec | std::ios_base::fixed |
		 std::ios_base::left);
	in >> utility::delim('(')
	   >> utility::delim(')');
	in.flags(flags);
	return in;
      }
      
    };
    
  private:
    param_type p;
    
  public:
    // constructor
    uniform01_dist() {
    }
    explicit uniform01_dist(const param_type &p) {
    }
    // reset internal state
    void reset() { }
    // random numbers
    template<typename R>
    result_type operator()(R &r) {
      return utility::uniformco<result_type>(r);
    }
    template<typename R>
    result_type operator()(R &r, const param_type &p) {
      return utility::uniformco<result_type>(r);
    }
    // property methods
    // min / max
    result_type min() const { return 0; }
    result_type max() const { return 1; }
    param_type param() const { return p; }
    void param(const param_type &p_new) { }
    // probability density function  
    result_type pdf(result_type x) const {
      if (x<0 or x>=1)
	return 0;
      return 1;
    }
    // cumulative density function 
    result_type cdf(result_type x) const {
      if (x<0)
	return 0;
      if (x>=1)
	return 1;
      return x;
    }
    // inverse cumulative density function 
    result_type icdf(result_type x) const {
      if (x<0 or x>1) {
	errno=EDOM;
	return math::numeric_limits<result_type>::quiet_NaN();
      }
      return x; 
    }
    
  };
  
  // -------------------------------------------------------------------

  // Equality comparable concept
  template<typename float_t>
  inline bool operator==(const typename uniform01_dist<float_t>::param_type &, 
			 const typename uniform01_dist<float_t>::param_type &) {
    return true;
  }

  template<typename float_t>
  inline bool operator!=(const typename uniform01_dist<float_t>::param_type &, 
			 const typename uniform01_dist<float_t>::param_type &) {
    return false;
  }
   
  // -------------------------------------------------------------------
  
  // Equality comparable concept
  template<typename float_t>
  inline bool operator==(const uniform01_dist<float_t> &g1, 
			 const uniform01_dist<float_t> &g2) {
    return g1.param()==g2.param();
  }

  template<typename float_t>
  inline bool operator!=(const uniform01_dist<float_t> &g1, 
			 const uniform01_dist<float_t> &g2) {
    return g1.param()!=g2.param();
  }
  
  // Streamable concept
  template<typename char_t, typename traits_t, typename float_t>
  std::basic_ostream<char_t, traits_t> &
  operator<<(std::basic_ostream<char_t, traits_t> &out,
	     const uniform01_dist<float_t> &g) {
    std::ios_base::fmtflags flags(out.flags());
    out.flags(std::ios_base::dec | std::ios_base::fixed |
	      std::ios_base::left);
    out << "[uniform01 " << g.param() << ']';
    out.flags(flags);
    return out;
  }
  
  template<typename char_t, typename traits_t, typename float_t>
  std::basic_istream<char_t, traits_t> &
  operator>>(std::basic_istream<char_t, traits_t> &in,
	     uniform01_dist<float_t> &g) {
    typename uniform01_dist<float_t>::param_type p;
    std::ios_base::fmtflags flags(in.flags());
    in.flags(std::ios_base::dec | std::ios_base::fixed |
	     std::ios_base::left);
    in >> utility::ignore_spaces()
       >> utility::delim("[uniform01 ") >> p >> utility::delim(']');
    if (in)
      g.param(p);
    in.flags(flags);
    return in;
  }
  
}

#endif
